function custom_layout_gui
% CUSTOM_LAYOUT_GUI — 2D canvas + 3D param editor + SDE/SDevice export
% - Build MV / L-wire / T-wire / Bus with a 2D canvas (grid 10x10).
% - 3D preview uses the same geometric model (stacked cuboids).
% - Export SDE writes explicit cuboids (SiO2, Ti, Gold, HfO2, Vacuum), already rotated & placed.
% - Any box with a non-empty "name" is treated as a CONTACT region (e.g., TopContact_south_1).
% - Export SDevice writes a solver deck that includes only the contacts that actually exist.
% - BottomContact is geometrically assigned in SDE by tagging the first Gold level.

%% ================== APP STATE ==================
gridN   = 10; cellSz  = 100; canvasW = gridN*cellSz; canvasH = gridN*cellSz;

Blocks = struct('type',{},'i',{},'j',{},'rot',{},'hRect',{},'hImg',{},'t',{});
SelectedIdx = [];
rotOptions  = [0 90 180 270]; rotPtr = 1; placingType = "";
ImgCache = containers.Map('KeyType','char','ValueType','any');

LastP = struct('L',24,'W',10,'H',2,'origin',[0 0 0], ...
    'H_Ti_bottom',1,'H_gold_bottom',2,'L_vac',4,'H_vac',3, ...
    'L_Au_top',16,'W_Au_top',8,'H_Au_top',4,'T_film',0.5,'CapSide_center',22);
NcellsArmDefault = 4;

%% ================== UI ==================
fig = uifigure('Name','Custom Layout GUI','Position',[50 50 1400 760],'Color',[0.97 0.98 1.0]);
main = uigridlayout(fig,[1 3]); main.ColumnWidth = {220,'1x',260}; main.RowHeight = {'1x'};
main.ColumnSpacing=10; main.Padding=[10 10 10 10];

left = uigridlayout(main,[8 1]); left.Layout.Column=1;
left.RowHeight = {30,70,70,70,70,'fit',30,30};
uilabel(left,'Text','Blocks','FontWeight','bold','FontSize',13);
uibutton(left,'Text','MV','FontWeight','bold','BackgroundColor',[1.0 0.95 0.7], 'ButtonPushedFcn',@(src,evt)setPlacing("MV"));
uibutton(left,'Text','L-wire','FontWeight','bold','BackgroundColor',[0.85 1.00 0.85], 'ButtonPushedFcn',@(src,evt)setPlacing("L-wire"));
uibutton(left,'Text','T-wire','FontWeight','bold','BackgroundColor',[0.85 0.92 1.00], 'ButtonPushedFcn',@(src,evt)setPlacing("T-wire"));
uibutton(left,'Text','Bus','FontWeight','bold','BackgroundColor',[1.00 0.85 0.85], 'ButtonPushedFcn',@(src,evt)setPlacing("bus"));

uilabel(left, ...
    'Text', ['It is recommended that users place the input stage of the desired layout on the left-hand side of the canvas and proceed toward the right, where the output stage should be positioned. ' ...
             'This left-to-right convention ensures consistent signal-flow directionality, facilitates intuitive circuit visualisation, and guarantees correct generation of the clock-zone phasing during subsequent 3D structure export and simulation.'], ...
    'FontSize',13,'WordWrap','on','HorizontalAlignment','left','BackgroundColor',[1.00 1.00 1.00]);

uilabel(left,'Text','Pick a block, then click on the canvas','FontSize',10,'FontColor',[0.3 0.3 0.4]);

centerPanel = uipanel(main,'Title','Canvas 10×10 (order & adjacency)','BackgroundColor',[0.98 0.98 1.0]); centerPanel.Layout.Column=2;
ax = uiaxes(centerPanel); ax.Units='normalized'; ax.Position=[0.05 0.05 0.9 0.9];
ax.XLim=[0 canvasW]; ax.YLim=[0 canvasH]; ax.DataAspectRatio=[1 1 1]; ax.Box='on'; ax.Toolbar.Visible='off'; hold(ax,'on');
title(ax,'Click to place blocks. 3D anchoring follows N/S/E/W neighbors.', 'FontSize',11);
drawGrid(ax,gridN,cellSz); ax.ButtonDownFcn=@(~,~)onCanvasClick();

right = uigridlayout(main,[14 1]); right.Layout.Column=3;
right.RowHeight = {40,40,40,40,20,40,40,20,40,40,40,'1x',20,20};
uibutton(right,'Text','Clear All','BackgroundColor',[1 1 1],'FontWeight','bold','ButtonPushedFcn',@(src,evt)clearAll());
uibutton(right,'Text','Delete Selected','BackgroundColor',[1 0.95 0.95],'ButtonPushedFcn',@(src,evt)deleteSelected());
btnRotate   = uibutton(right,'Text','Rotate (0°)','BackgroundColor',[0.95 0.97 1.0],'ButtonPushedFcn',@(src,evt)rotateSelected());
uibutton(right,'Text','Build 3D','BackgroundColor',[0.15 0.45 0.85],'FontWeight','bold','ButtonPushedFcn',@(src,evt)build3D(),'FontColor',[1 1 1]);
uilabel(right,'Text','Click a cell to select','FontSize',10,'FontColor',[0.3 0.3 0.4]);
lblInfo = uilabel(right,'Text','0 blocks','FontSize',11);
lblSel  = uilabel(right,'Text','Selected: none','FontSize',11);
uilabel(right,'Text',''); % spacer
uibutton(right,'Text','Export SDE.txt','BackgroundColor',[0.10 0.55 0.25],'FontWeight','bold','FontColor',[1 1 1], ...
    'Tooltip','Generate an SDE deck (MV/L/T/Bus) with positions & rotations from the canvas', ...
    'ButtonPushedFcn',@(src,evt)exportSDE_btn());
uibutton(right,'Text','Export SDevice.txt','BackgroundColor',[0.10 0.35 0.55],'FontWeight','bold','FontColor',[1 1 1], ...
    'Tooltip','Generate an SDevice deck using detected contacts', ...
    'ButtonPushedFcn',@(src,evt)exportSDevice_btn());

%% ================== Canvas callbacks ==================
    function setPlacing(tp), placingType=tp; fig.Pointer='crosshair'; end
    function onCanvasClick()
        cp=ax.CurrentPoint(1,1:2); x=cp(1); y=cp(2); if x<0||x>canvasW||y<0||y>canvasH, return; end
        i=floor(x/cellSz)+1; if i>gridN, i=gridN; end; j=floor(y/cellSz)+1; if j>gridN, j=gridN; end
        if placingType~="", addBlock(placingType,i,j); placingType=""; fig.Pointer='arrow'; return; end
        selectByCell(i,j);
    end
    function addBlock(tp,i,j)
        idx=findBlockAt(i,j); if ~isempty(idx), doDeleteBlock(idx); end
        [xc,yc]=cellLowerLeft(i,j,cellSz);
        hr=rectangle(ax,'Position',[xc yc cellSz cellSz],'FaceColor',colorForType(tp),'EdgeColor',[0.2 0.2 0.2],'LineWidth',1.2,'ButtonDownFcn',@(src,evt)selectByCell(i,j));
        hImg=drawPNG(ax,tp,[xc yc],cellSz,rotOptions(rotPtr));
        B=struct('type',char(tp),'i',i,'j',j,'rot',rotOptions(rotPtr),'hRect',hr,'hImg',hImg,'t',now);
        Blocks(end+1)=B; SelectedIdx=numel(Blocks); updateLabels(); updateRotateLabel(); highlightSelection();
    end
    function selectByCell(i,j)
        idx=findBlockAt(i,j); if isempty(idx), SelectedIdx=[]; highlightSelection(); updateLabels(); return; end
        SelectedIdx=idx; highlightSelection(); updateLabels(); updateRotateLabel();
    end
    function idx=findBlockAt(i,j)
        idx=[]; for k=1:numel(Blocks), if Blocks(k).i==i && Blocks(k).j==j, idx=k; return; end, end
    end
    function deleteSelected(), if isempty(SelectedIdx), return; end, doDeleteBlock(SelectedIdx); SelectedIdx=[]; updateLabels(); end
    function doDeleteBlock(k)
        if k<1||k>numel(Blocks), return; end
        if isfield(Blocks(k),'hRect')&&isvalid(Blocks(k).hRect), delete(Blocks(k).hRect); end
        if isfield(Blocks(k),'hImg') &&isvalid(Blocks(k).hImg),  delete(Blocks(k).hImg);  end
        Blocks(k)=[]; if ~isempty(SelectedIdx)&&SelectedIdx>numel(Blocks), SelectedIdx=[]; end; highlightSelection();
    end
    function clearAll()
        for k=1:numel(Blocks)
            if isfield(Blocks(k),'hRect')&&isvalid(Blocks(k).hRect), delete(Blocks(k).hRect); end
            if isfield(Blocks(k),'hImg') &&isvalid(Blocks(k).hImg),  delete(Blocks(k).hImg);  end
        end
        Blocks=struct('type',{},'i',{},'j',{},'rot',{},'hRect',{},'hImg',{},'t',{});
        SelectedIdx=[]; updateLabels();
    end
    function rotateSelected()
        if isempty(SelectedIdx), return; end
        rotPtr=mod(rotPtr,numel(rotOptions))+1; Blocks(SelectedIdx).rot=rotOptions(rotPtr);
        updateRotateLabel(); highlightSelection(); applyRotation2D(SelectedIdx);
    end
    function applyRotation2D(k)
        b=Blocks(k); if ~isempty(b.hImg)&&isvalid(b.hImg), delete(b.hImg); end
        [xc,yc]=cellLowerLeft(b.i,b.j,cellSz); Blocks(k).hImg=drawPNG(ax,b.type,[xc yc],cellSz,b.rot);
    end
    function highlightSelection()
        for kk=1:numel(Blocks)
            if ~isempty(Blocks(kk).hRect)&&isvalid(Blocks(kk).hRect)
                Blocks(kk).hRect.LineWidth=1.2; Blocks(kk).hRect.EdgeColor=[0.2 0.2 0.2];
            end
        end
        if ~isempty(SelectedIdx)&&SelectedIdx>=1&&SelectedIdx<=numel(Blocks)
            if ~isempty(Blocks(SelectedIdx).hRect)&&isvalid(Blocks(SelectedIdx).hRect)
                Blocks(SelectedIdx).hRect.LineWidth=2.5; Blocks(SelectedIdx).hRect.EdgeColor=[0.0 0.4 0.85];
            end
        end
    end
    function updateRotateLabel()
        if isempty(SelectedIdx), btnRotate.Text=sprintf('Rotate (%d°)',rotOptions(rotPtr));
        else, b=Blocks(SelectedIdx); btnRotate.Text=sprintf('Rotate (%d°) — current: %d°',rotOptions(rotPtr),b.rot);
        end
    end
    function updateLabels()
        lblInfo.Text=sprintf('%d blocks',numel(Blocks));
        if isempty(SelectedIdx), lblSel.Text='Selected: none';
        else, b=Blocks(SelectedIdx); lblSel.Text=sprintf('Selected: %s @ (%d,%d), rot=%d°',b.type,b.i,b.j,b.rot);
        end
    end
    function [x0,y0]=cellLowerLeft(i,j,csz), x0=(i-1)*csz; y0=(j-1)*csz; end
    function col=colorForType(tp)
        switch lower(tp)
            case 'mv', col=[1.0 0.95 0.7];
            case 'l-wire', col=[0.85 1.00 0.85];
            case 't-wire', col=[0.85 0.92 1.00];
            case 'bus', col=[1.00 0.85 0.85];
            otherwise, col=[0.95 0.95 0.95];
        end
    end
    function drawGrid(axh,n,csz)
        for xi=0:n, x=xi*csz; line(axh,[x x],[0 n*csz],'Color',[0.85 0.88 0.93],'LineWidth',1,'HitTest','off'); end
        for yi=0:n, y=yi*csz; line(axh,[0 n*csz],[y y],'Color',[0.85 0.88 0.93],'LineWidth',1,'HitTest','off'); end
        rectangle(axh,'Position',[0 0 n*csz n*csz],'EdgeColor',[0.4 0.45 0.55],'LineWidth',1.5,'HitTest','off');
    end

%% ================== PNG Icons ==================
    function hIm=drawPNG(axh,tp,LL,csz,rotdeg)
        base=getBaseImage(lower(tp)); C=base.C; A=base.A; k=mod(round(rotdeg/90),4);
        if k~=0, C=rot90(C,-k); if ~isempty(A), A=rot90(A,-k); end, end
        x=LL(1); y=LL(2); hIm=image(axh,'XData',[x x+csz],'YData',[y y+csz],'CData',C);
        if ~isempty(A), set(hIm,'AlphaData',A); end; uistack(hIm,'top');
        i=floor(x/csz)+1; j=floor(y/csz)+1; hIm.ButtonDownFcn=@(~,~)selectByCell(i,j);
    end
    function base=getBaseImage(tpLower)
        if isKey(ImgCache,tpLower), base=ImgCache(tpLower); return; end
        switch tpLower
            case 'mv', fname='mv.png';
            case 'l-wire', fname='lwire.png';
            case 't-wire', fname='twire.png';
            case 'bus', fname='bus.png';
            otherwise, fname='';
        end
        C=[]; A=[]; if ~isempty(fname)&&exist(fname,'file'), [C,~,A]=imread(fname); end
        base=struct('C',C,'A',A); ImgCache(tpLower)=base;
    end

%% ================== 3D Editor ==================
    function build3D()
        if isempty(Blocks), uialert(fig,'No blocks on canvas.','Build 3D','Icon','warning'); return; end
        [~,ord]=sort([Blocks.t]); Bseq=Blocks(ord);
        open3DEditor(Bseq,LastP);
    end

    function open3DEditor(Bseq,Pinit)
        S = struct();
        S.f = uifigure('Name','3D Editor','Position',[120 120 1240 740],'Color',[1 1 1]);
        gl = uigridlayout(S.f,[1 2]); gl.ColumnWidth={360,'1x'};
        ctl=uigridlayout(gl,[22 2]); ctl.ColumnWidth={170,170}; ctl.Padding=[10 10 10 10]; ctl.RowSpacing=6; ctl.ColumnSpacing=8;
        axp=uipanel(gl,'Title','3D Preview'); S.ax3=axes(axp); S.ax3.Toolbar.Visible='off';
        hold(S.ax3,'on'); grid(S.ax3,'on'); axis(S.ax3,'equal'); view(S.ax3,35,25); xlabel(S.ax3,'X'); ylabel(S.ax3,'Y'); zlabel(S.ax3,'Z');
        S.Bseq=Bseq; S.P=Pinit; S.NcellsArm=NcellsArmDefault; S.sceneHandles=gobjects(0); S.ctlMap=struct();

        addNumField('L',S.P.L); addNumField('W',S.P.W); addNumField('H',S.P.H);
        addNumField('H_Ti_bottom',S.P.H_Ti_bottom); addNumField('H_gold_bottom',S.P.H_gold_bottom);
        addNumField('H_Au_top',S.P.H_Au_top,'H_Au_top',0); addNumField('H_vac',S.P.H_vac,'H_vac',0);
        addNumField('L_vac',min(S.P.L_vac,S.P.L),'L_vac',0,S.P.L);
        tf_max=min(S.P.L,S.P.W)/2; addNumField('T_film',min(S.P.T_film,tf_max),'T_film',0,tf_max);
        [laumax,waumax]=auMaxima(); addNumField('L_Au_top',min(S.P.L_Au_top,laumax),'L_Au_top',0,laumax);
        addNumField('W_Au_top',min(S.P.W_Au_top,waumax),'W_Au_top',0,waumax);

        btnExportSDE = uibutton(ctl,'Text','Export SDE.txt', ...
            'ButtonPushedFcn',@(src,evt)exportSDE_btn(), ...
            'BackgroundColor',[0.10 0.55 0.25],'FontColor',[1 1 1],'FontWeight','bold'); btnExportSDE.Layout.Column=[1 2];

        btnExportSDev = uibutton(ctl,'Text','Export SDevice.txt', ...
            'ButtonPushedFcn',@(src,evt)exportSDevice_btn(), ...
            'BackgroundColor',[0.10 0.35 0.55],'FontColor',[1 1 1],'FontWeight','bold'); btnExportSDev.Layout.Column=[1 2];

        uilabel(ctl,'Text',''); uilabel(ctl,'Text','');
        uibutton(ctl,'Text','Reset','ButtonPushedFcn',@(src,evt)doReset()); uibutton(ctl,'Text','Close','ButtonPushedFcn',@(src,evt)close(S.f));

        clampAll(); updateDependentLimits(); rerender();

        function addNumField(lbl,val,fieldName,minv,maxv)
            if nargin<3||isempty(fieldName), fieldName=lbl; end
            if nargin<4, minv=-Inf; end, hasMax=(nargin>=5)&&~isempty(maxv);
            uilabel(ctl,'Text',lbl,'HorizontalAlignment','right');
            ef=uieditfield(ctl,'numeric','Value',val); if hasMax, ef.Limits=[minv,maxv]; else, ef.Limits=[minv,Inf]; end
            ef.ValueDisplayFormat='%.6g'; ef.ValueChangedFcn=@(h,~)onChange(fieldName,h.Value); S.ctlMap.(fieldName)=ef;
        end
        function [laumax,waumax]=auMaxima(), laumax=max(0,S.P.L-2*S.P.T_film); waumax=max(0,S.P.W-2*S.P.T_film); end
        function clampAll()
            tf_max=min(S.P.L,S.P.W)/2; S.P.T_film=min(max(0,S.P.T_film),tf_max);
            S.P.L_vac=min(max(0,S.P.L_vac),S.P.L);
            [laumax,waumax]=auMaxima(); S.P.L_Au_top=min(max(0,S.P.L_Au_top),laumax); S.P.W_Au_top=min(max(0,S.P.W_Au_top),waumax);
            S.P.H=max(0,S.P.H); S.P.H_Ti_bottom=max(0,S.P.H_Ti_bottom); S.P.H_gold_bottom=max(0,S.P.H_gold_bottom);
            S.P.H_vac=max(0,S.P.H_vac); S.P.H_Au_top=max(0,S.P.H_Au_top);
        end
        function updateDependentLimits()
            tf_max=min(S.P.L,S.P.W)/2; if isfield(S.ctlMap,'T_film'), S.ctlMap.T_film.Limits=[0,tf_max]; end
            if S.P.T_film>tf_max, S.P.T_film=tf_max; if isfield(S.ctlMap,'T_film'), S.ctlMap.T_film.Value=S.P.T_film; end, end
            if isfield(S.ctlMap,'L_vac'), S.ctlMap.L_vac.Limits=[0,S.P.L]; if S.P.L_vac>S.P.L, S.P.L_vac=S.P.L; S.ctlMap.L_vac.Value=S.P.L_vac; end, end
            [laumax,waumax]=auMaxima();
            if isfield(S.ctlMap,'L_Au_top'), S.ctlMap.L_Au_top.Limits=[0,laumax]; if S.P.L_Au_top>laumax, S.P.L_Au_top=laumax; S.ctlMap.L_Au_top.Value=S.P.L_Au_top; end, end
            if isfield(S.ctlMap,'W_Au_top'), S.ctlMap.W_Au_top.Limits=[0,waumax]; if S.P.W_Au_top>waumax, S.P.W_Au_top=waumax; S.ctlMap.W_Au_top.Value=S.P.W_Au_top; end, end
        end
        function onChange(fname,v)
            if isfield(S.P,fname), S.P.(fname)=v; end, LastP=S.P; clampAll(); updateDependentLimits(); rerender();
        end
        function doReset(), S.P=Pinit; LastP=S.P; clampAll(); fns=fieldnames(S.ctlMap);
            for ii=1:numel(fns), fn=fns{ii}; if isfield(S.P,fn), S.ctlMap.(fn).Value=S.P.(fn); end, end
            updateDependentLimits(); rerender();
        end
        function rerender()
            busX=4*max(0,S.P.W);
            if ~isempty(S.sceneHandles), try delete(S.sceneHandles(ishandle(S.sceneHandles))); catch, end, end
            S.sceneHandles=render3DScene(S.ax3,S.Bseq,S.P,S.NcellsArm,busX);
            camlight(S.ax3,'headlight'); lighting(S.ax3,'flat');
        end
    end

%% ================== Export SDE ==================
    function exportSDE_btn()
        if isempty(Blocks), uialert(fig,'No blocks on canvas to export.','Export SDE','Icon','warning'); return; end
        [fn,fp]=uiputfile('*.txt','Save SDE deck as...'); if isequal(fn,0), return; end, outFile=fullfile(fp,fn);

        [~,ord]=sort([Blocks.t]); BseqLocal=Blocks(ord);
        busX=4*max(0,LastP.W);
        Poses=computeCanvasPosesLikeRenderer(BseqLocal,LastP,NcellsArmDefault,busX);

        fid=fopen(outFile,'w'); if fid<0, uialert(fig,'Cannot open output file.','Export SDE','Icon','error'); return; end
        cleaner=onCleanup(@()fclose(fid)); 

        emitHeader(fid);
        emitGlobalParams(fid,LastP,NcellsArmDefault);
        fprintf(fid,';; ================== GEOMETRY (from canvas, rotated & placed) ==================\n');

        ContactMap = containers.Map('KeyType','char','ValueType','any');
        bottomAuVars = {};            
        au_counter = 0; reg_counter = 0;
        tol = 1e-9;

        for k=1:numel(Poses)
            pb=Poses(k); t=lower(string(pb.type)); rot=mod(pb.rot,360); xLL=pb.LL(1); yLL=pb.LL(2);
            fprintf(fid,'\n;; ---- BLOCK #%d  %s  rot=%d°  LL=(%.6g, %.6g)\n',k,upper(char(t)),rot,xLL,yLL);

            [boxes,nativeLL,nativeSize]=buildBlockBoxesNative(t,LastP,NcellsArmDefault,busX);
            Cn=nativeLL+nativeSize/2;
            boxesR=rotateBoxesAround(boxes,Cn,rot);
            [minxyR,~]=boxesAABB(boxesR); delta=[xLL-minxyR(1), yLL-minxyR(2)];
            boxesT=translateBoxesXY(boxesR,delta);

            for ii=1:numel(boxesT)
                V=boxesT(ii).V; x1=min(V(:,1)); y1=min(V(:,2)); z1=min(V(:,3));
                x2=max(V(:,1)); y2=max(V(:,2)); z2=max(V(:,3));
                if (x2-x1)<=0||(y2-y1)<=0||(z2-z1)<=0, continue; end
                mat=char(boxesT(ii).mat); cname=strtrim(char(boxesT(ii).name));
                isBottomAu = strcmp(mat,'Gold') && ...
                             abs((z2-z1) - LastP.H_gold_bottom) < tol && ...
                             abs(z1 - (LastP.H + LastP.H_Ti_bottom)) < tol;

                if ~isempty(cname)
                    au_counter = au_counter + 1;
                    varName = sprintf('CNT_%03d', au_counter);
                    fprintf(fid,'(define %s (mk-cuboid %.9g %.9g %.9g %.9g %.9g %.9g "%s"))\n', ...
                        varName, x1,y1,z1, x2,y2,z2, mat);
                    if ~isKey(ContactMap,cname), ContactMap(cname) = {varName};
                    else, ContactMap(cname) = [ContactMap(cname), {varName}]; end
                else
                    reg_counter = reg_counter + 1;
                    varName = sprintf('REG_%03d', reg_counter);
                    fprintf(fid,'(define %s (mk-cuboid %.9g %.9g %.9g %.9g %.9g %.9g "%s"))\n', ...
                        varName, x1,y1,z1, x2,y2,z2, mat);
                    if isBottomAu
                        bottomAuVars{end+1} = varName; 
                    end
                end
            end
        end

        fprintf(fid,'\n;; ================== CONTACTS (auto-detected) ==================\n');

        fprintf(fid,'(sdegeo:define-contact-set "BottomContact" 4 (color:rgb 0.1 0.1 0.1) "##")\n');
        fprintf(fid,'(sdegeo:set-current-contact-set "BottomContact")\n');
        for v=1:numel(bottomAuVars)
            fprintf(fid,'(sdegeo:set-contact %s "BottomContact")\n', bottomAuVars{v});
        end
        if isempty(bottomAuVars)
            fprintf(fid,';; [INFO] No first-level Gold regions detected for BottomContact.\n');
        end

        emitContactSetsAndAssignments(fid, ContactMap);

        fprintf(fid,'\n;; ================== MESH & BUILD ==================\n');
        emitMeshRefAndBuild(fid);

        uialert(fig,sprintf('SDE exported:\n%s',outFile),'Export SDE','Icon','success');
    end

%% ================== Export SDevice ==================
    function exportSDevice_btn()
        if isempty(Blocks), uialert(fig,'No blocks on canvas to export.','Export SDevice','Icon','warning'); return; end
        [fn,fp]=uiputfile('*.txt','Save SDevice deck as...'); if isequal(fn,0), return; end, outFile=fullfile(fp,fn);

        [~,ord]=sort([Blocks.t]); BseqLocal=Blocks(ord);
        busX=4*max(0,LastP.W);
        Poses=computeCanvasPosesLikeRenderer(BseqLocal,LastP,NcellsArmDefault,busX);

        ContactNames = collectContactNames(Poses, LastP, NcellsArmDefault, busX);

        fid=fopen(outFile,'w'); if fid<0, uialert(fig,'Cannot open output file.','Export SDevice','Icon','error'); return; end
        cleaner=onCleanup(@()fclose(fid)); 

        emitSDevice(fid, ContactNames);

        uialert(fig,sprintf('SDevice exported:\n%s',outFile),'Export SDevice','Icon','success');
    end

%% ================== 3D SCENE & GEOMETRY HELPERS ==================
    function HAll=render3DScene(ax3,Bseq,P,NcellsArm,BusFillX)
        hold(ax3,'on'); grid(ax3,'on'); axis(ax3,'equal'); view(ax3,35,25);
        title(ax3,'3D — face-anchored packing according to canvas neighbors');

        placed=struct('LL',{},'w',{},'hgt',{},'rot',{}); HAll=gobjects(0);
        for k=1:numel(Bseq)
            b=Bseq(k); [w0,h0]=nativeBlockSize(b,P,NcellsArm,BusFillX);
            if mod(b.rot,180)==0, wnat=w0; hnat=h0; else, wnat=h0; hnat=w0; end
            [hasAnchor,aIdx,side]=findAnchorNeighbor(Bseq,placed,k);
            if ~hasAnchor, LL=[0,0];
            else
                prev=placed(aIdx);
                switch side
                    case 'E', LL=[prev.LL(1)+prev.w, prev.LL(2)];
                    case 'W', LL=[prev.LL(1)-wnat,    prev.LL(2)];
                    case 'N', LL=[prev.LL(1),         prev.LL(2)+prev.hgt];
                    case 'S', LL=[prev.LL(1),         prev.LL(2)-hnat];
                    otherwise, LL=[prev.LL(1)+prev.w, prev.LL(2)];
                end
            end
            [boxes,nativeLL,nativeSize]=buildBlockBoxesNative(lower(b.type),P,NcellsArm,BusFillX);
            Cn=nativeLL+nativeSize/2; boxesR=rotateBoxesAround(boxes,Cn,b.rot);
            [minxyR,~]=boxesAABB(boxesR); delta=[LL(1)-minxyR(1), LL(2)-minxyR(2)];
            boxesT=translateBoxesXY(boxesR,delta);
            H=drawBoxes(ax3,boxesT);
            [~,maxxy2]=boxesAABB(boxesT); 
            placed(k).LL=LL; placed(k).w=wnat; placed(k).hgt=hnat; placed(k).rot=b.rot;
            HAll=[HAll; H]; %#ok<AGROW>
        end
        if ~isempty(HAll)&&all(ishandle(HAll))
            Vall=[]; for hh=HAll(:).', try V=get(hh,'Vertices'); if ~isempty(V), Vall=[Vall;V]; end, catch, end, end %#ok<AGROW>
            if ~isempty(Vall)
                xl=[min(Vall(:,1)) max(Vall(:,1))]; yl=[min(Vall(:,2)) max(Vall(:,2))]; zl=[min(Vall(:,3)) max(Vall(:,3))];
                dx=diff(xl); dy=diff(yl); dz=diff(zl); pad=0.05*max([dx dy max(dz,1)]);
                xlim(ax3,[xl(1)-pad xl(2)+pad]); ylim(ax3,[yl(1)-pad yl(2)+pad]); zlim(ax3,[max(0,zl(1)-pad) zl(2)+pad]);
            end
        end
    end

    function [hasAnchor,aIdx,side]=findAnchorNeighbor(Bseq,placed,k)
        hasAnchor=false; aIdx=[]; side='E'; if k==1, return; end
        b=Bseq(k); bestIdx=0; bestTime=-inf; bestSide='E';
        for j=1:k-1
            bj=Bseq(j); di=b.i-bj.i; dj=b.j-bj.j; if abs(di)+abs(dj)~=1, continue; end
            if di==1, s='E'; elseif di==-1, s='W'; elseif dj==1, s='N'; else, s='S'; end
            if bj.t>bestTime, bestTime=bj.t; bestIdx=j; bestSide=s; end
        end
        if bestIdx>0, hasAnchor=true; aIdx=bestIdx; side=bestSide; end
    end

    function [w0,h0]=nativeBlockSize(b,P,NcellsArm,BusFillX)
        arm_span=NcellsArm*P.W; total_span=2*arm_span+P.L;
        switch lower(b.type)
            case 'mv',     w0=total_span; h0=total_span;
            case 'l-wire', w0=P.L+arm_span; h0=arm_span+P.L+arm_span;
            case 't-wire', w0=total_span; h0=total_span;
            case 'bus',    w0=P.L+2*BusFillX; h0=3*P.W;
            otherwise,     w0=100; h0=100;
        end
    end

%% ================== Block → Boxes (native) ==================
    function [boxes,LL,SZ]=buildBlockBoxesNative(tp,P,NcellsArm,BusFillX)
        switch tp
            case 'mv',     [boxes,LL,SZ]=boxesMV(P,NcellsArm);
            case 'l-wire', [boxes,LL,SZ]=boxesLwire(P,NcellsArm);
            case 't-wire', [boxes,LL,SZ]=boxesTwire(P,NcellsArm);
            case 'bus',    [boxes,LL,SZ]=boxesBus(P,3,BusFillX);
            otherwise,     boxes=makeBoxList([0 0 0],[P.L P.W P.H],"SiO2",""); [LL,SZ]=boxesAABB(boxes); SZ=SZ-LL;
        end
    end

    function [boxes,LL,SZ]=boxesMV(P,NcellsArm)
        boxes=[]; pitch=P.W; x0=NcellsArm*P.W; y0=NcellsArm*P.W;

        % SOUTH arm (MV input, 1 = far, 4 = near)
        y_cursor=0;
        for i=1:NcellsArm
            idx = i;
            cname = sprintf("TopContact_MV_IN_%d", idx);
            boxes=[boxes; boxesCellMV(P,[x0 y0+y_cursor 0],cname)]; %#ok<AGROW>
            y_cursor=y_cursor+pitch;
        end
        % Center (MV)
        yB=y0+y_cursor; boxes=[boxes; boxesCentralSquareMV(P,[x0 yB 0],"TopContact_MV_X")]; %#ok<AGROW>
        % NORTH arm (MV input, flip index so 4 = near center)
        yB2=yB+P.L; y_cursor=0;
        for i=1:NcellsArm
            idx = NcellsArm - i + 1;
            cname = sprintf("TopContact_MV_IN_%d", idx);
            boxes=[boxes; boxesCellMV(P,[x0 yB2+y_cursor 0],cname)]; %#ok<AGROW>
            y_cursor=y_cursor+pitch;
        end
        % WEST arm
        Crot=[x0+P.L/2, yB+P.L/2]; boxesSouthForRot=[]; y_cursor=0;
        for i=1:NcellsArm
            idx = i;
            cname = sprintf("TopContact_MV_IN_%d", idx);
            tmp=boxesCellMV(P,[x0 y0+y_cursor 0],cname);
            boxesSouthForRot=[boxesSouthForRot; tmp]; y_cursor=y_cursor+pitch; %#ok<AGROW>
        end
        boxesWest=rotateBoxesAround(boxesSouthForRot,Crot,90);
        % EAST arm (MV output, 1 = near, 4 = far)
        boxesNorthForRot=[]; y_cursor=0;
        for i=1:NcellsArm
            idx = i;
            cname = sprintf("TopContact_MV_OUT_%d", idx);
            tmp=boxesCellMV(P,[x0 yB2+y_cursor 0],cname);
            boxesNorthForRot=[boxesNorthForRot; tmp]; y_cursor=y_cursor+pitch; %#ok<AGROW>
        end
        boxesEast=rotateBoxesAround(boxesNorthForRot,Crot,90);

        boxes=[boxes; boxesWest; boxesEast];

        arm_span=NcellsArm*pitch; heights=[P.H+P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top];
        boxes=[boxes; boxesFiller([x0+P.L yB+P.L 0],arm_span,arm_span,heights)];
        boxes=[boxes; boxesFiller([x0-arm_span yB+P.L 0],arm_span,arm_span,heights)];
        boxes=[boxes; boxesFiller([x0-arm_span yB-arm_span 0],arm_span,arm_span,heights)];
        boxes=[boxes; boxesFiller([x0+P.L yB-arm_span 0],arm_span,arm_span,heights)];

        [LL,SZ]=boxesAABB(boxes); SZ=SZ-LL;
    end

    function [boxes,LL,SZ]=boxesLwire(P,NcellsArm)
        boxes=[]; pitch=P.W; x0=NcellsArm*P.W; y0=NcellsArm*P.W;

        % NORTH arm (L input, 1 = far, 4 = near)
        y_cursor=0;
        for i=1:NcellsArm
            idx = i;
            cname = sprintf("TopContact_L_IN_%d", idx);
            boxes=[boxes; boxesCellMV(P,[x0 y0+y_cursor 0],cname)]; %#ok<AGROW>
            y_cursor=y_cursor+pitch;
        end
        % Center (L)
        yB=y0+y_cursor; boxes=[boxes; boxesCentralSquareMV(P,[x0 yB 0],"TopContact_L_X")]; %#ok<AGROW>
        % EAST arm along X (L output, 1 = near, 4 = far)
        x_cursor=x0+P.L;
        for i=1:NcellsArm
            idx = i;
            cname = sprintf("TopContact_L_OUT_%d", idx);
            boxes=[boxes; boxesCellLX(P,[x_cursor yB 0],cname)]; %#ok<AGROW>
            x_cursor=x_cursor+pitch;
        end

        arm_span=NcellsArm*pitch; yB2=yB+P.L; heights=[P.H+P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top];
        boxes=[boxes; boxesFiller([x0+P.L yB2 0],arm_span,arm_span,heights)];
        boxes=[boxes; boxesFiller([x0-arm_span yB2 0],arm_span,arm_span,heights)];
        boxes=[boxes; boxesFiller([x0-arm_span yB-arm_span 0],arm_span,arm_span,heights)];
        boxes=[boxes; boxesFiller([x0+P.L yB-arm_span 0],arm_span,arm_span,heights)];
        boxes=[boxes; boxesFiller([x0-arm_span yB 0],arm_span,P.L,heights)];
        boxes=[boxes; boxesFiller([x0 yB+P.L 0],P.L,arm_span,heights)];

        [LL,SZ]=boxesAABB(boxes); SZ=SZ-LL;
    end

    function [boxes,LL,SZ]=boxesTwire(P,NcellsArm)
        boxes=[]; pitch=P.W; x0=NcellsArm*P.W; y0=NcellsArm*P.W;

        % NORTH arm (T input, 1 = far, 4 = near)
        y_cursor=0;
        for i=1:NcellsArm
            idx = i;
            cname = sprintf("TopContact_T_IN_%d", idx);
            boxes=[boxes; boxesCellMV(P,[x0 y0+y_cursor 0],cname)]; %#ok<AGROW>
            y_cursor=y_cursor+pitch;
        end
        % Center (T)
        yB=y0+y_cursor; boxes=[boxes; boxesCentralSquareMV(P,[x0 yB 0],"TopContact_T_X")]; %#ok<AGROW>
        % SOUTH arm (T input, flipped index so 4 = near to center)
        y_cursor=yB+P.L;
        for i=1:NcellsArm
            idx = NcellsArm - i + 1;
            cname = sprintf("TopContact_T_IN_%d", idx);
            boxes=[boxes; boxesCellMV(P,[x0 y_cursor 0],cname)]; %#ok<AGROW>
            y_cursor=y_cursor+pitch;
        end
        % EAST arm along X (T output, 1 = near, 4 = far)
        x_cursor=x0+P.L;
        for i=1:NcellsArm
            idx = i;
            cname = sprintf("TopContact_T_OUT_%d", idx);
            boxes=[boxes; boxesCellLX(P,[x_cursor yB 0],cname)]; %#ok<AGROW>
            x_cursor=x_cursor+pitch;
        end

        arm_span=NcellsArm*pitch; yB2=yB+P.L; heights=[P.H+P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top];
        boxes=[boxes; boxesFiller([x0+P.L yB2 0],arm_span,arm_span,heights)];
        boxes=[boxes; boxesFiller([x0-arm_span yB2 0],arm_span,arm_span,heights)];
        boxes=[boxes; boxesFiller([x0-arm_span yB-arm_span 0],arm_span,arm_span,heights)];
        boxes=[boxes; boxesFiller([x0+P.L yB-arm_span 0],arm_span,arm_span,heights)];
        boxes=[boxes; boxesFiller([x0-arm_span yB 0],arm_span,P.L,heights)];

        [LL,SZ]=boxesAABB(boxes); SZ=SZ-LL;
    end

    function [boxes,LL,SZ]=boxesBus(P,Ncells,BusFillX)
        boxes=[]; x0=BusFillX; y0=0;
        for i=1:Ncells
            cname = sprintf("TopContact_b_%d", i);
            boxes=[boxes; boxesCellMV(P,[x0 y0+(i-1)*P.W 0],cname)]; %#ok<AGROW>
        end
        heights=[P.H+P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top];
        LxFill=max(0,x0); WyFill=max(Ncells*P.W,30);
        boxes=[boxes; boxesFiller([x0-LxFill,y0,0],LxFill,WyFill,heights)];
        boxes=[boxes; boxesFiller([x0+P.L,   y0,0],LxFill,WyFill,heights)];
        [LL,SZ]=boxesAABB(boxes); SZ=SZ-LL;
    end

%% ================== Atomics ==================
    function bx=boxesCellMV(P,o,contactName)
        bx=[];
        bx=[bx; makeBoxList(o+[0 0 0],[P.L P.W P.H],"SiO2","")];
        bx=[bx; makeBoxList(o+[0 0 P.H],[P.L P.W P.H_Ti_bottom],"Titanium","")];
        bx=[bx; makeBoxList(o+[0 0 P.H+P.H_Ti_bottom],[P.L P.W P.H_gold_bottom],"Gold","")];
        z0_top=o(3)+P.H+P.H_Ti_bottom+P.H_gold_bottom;
        L_vac_eff=min(max(P.L_vac,0),P.L); L_left=max(0,(P.L-L_vac_eff)/2); L_right=max(0,P.L-(L_left+L_vac_eff));
        if L_left>0,  bx=[bx; makeBoxList([o(1) o(2) z0_top],[L_left P.W P.H_vac],"HfO2","")]; end
        if L_vac_eff>0, bx=[bx; makeBoxList([o(1)+L_left o(2) z0_top],[L_vac_eff P.W P.H_vac],"Vacuum","")]; end
        if L_right>0, bx=[bx; makeBoxList([o(1)+L_left+L_vac_eff o(2) z0_top],[L_right P.W P.H_vac],"HfO2","")]; end
        x0_cap=o(1)+(P.L-P.L_Au_top)/2; y0_cap=o(2)+(P.W-P.W_Au_top)/2; z0_cap=z0_top+P.H_vac;
        if P.L_Au_top>0 && P.W_Au_top>0 && P.H_Au_top>0
            bx=[bx; makeBoxList([x0_cap y0_cap z0_cap],[P.L_Au_top P.W_Au_top P.H_Au_top],"Gold",contactName)];
        end
        if P.T_film>0
            bx=[bx; makeBoxList([x0_cap-P.T_film, y0_cap, z0_cap],[P.T_film, P.W_Au_top, P.H_Au_top],"Titanium","")];
            bx=[bx; makeBoxList([x0_cap+P.L_Au_top, y0_cap, z0_cap],[P.T_film, P.W_Au_top, P.H_Au_top],"Titanium","")];
            bx=[bx; makeBoxList([x0_cap-P.T_film, y0_cap-P.T_film, z0_cap],[P.L_Au_top+2*P.T_film, P.T_film, P.H_Au_top],"Titanium","")];
            bx=[bx; makeBoxList([x0_cap-P.T_film, y0_cap+P.W_Au_top, z0_cap],[P.L_Au_top+2*P.T_film, P.T_film, P.H_Au_top],"Titanium","")];
        end
        x_in_min=x0_cap-P.T_film; x_in_max=x0_cap+P.L_Au_top+P.T_film;
        y_in_min=y0_cap-P.T_film; y_in_max=y0_cap+P.W_Au_top+P.T_film;
        T_left_frame =max(0,(x_in_min-o(1)));
        T_right_frame=max(0,(o(1)+P.L-x_in_max));
        T_front_frame=max(0,(y_in_min-o(2)));
        T_back_frame =max(0,(o(2)+P.W-y_in_max));
        z0_frame=z0_cap;
        if T_left_frame>0,  bx=[bx; makeBoxList([o(1) o(2) z0_frame],[T_left_frame  P.W P.H_Au_top],"SiO2","")]; end
        if T_right_frame>0, bx=[bx; makeBoxList([x_in_max o(2) z0_frame],[T_right_frame P.W P.H_Au_top],"SiO2","")]; end
        if T_front_frame>0, bx=[bx; makeBoxList([x_in_min o(2) z0_frame],[max(0,x_in_max-x_in_min) T_front_frame P.H_Au_top],"SiO2","")]; end
        if T_back_frame>0,  bx=[bx; makeBoxList([x_in_min y_in_max z0_frame],[max(0,x_in_max-x_in_min) T_back_frame P.H_Au_top],"SiO2","")]; end
    end

    function bx=boxesCellLX(P,o,contactName)
        bx=[];
        bx=[bx; makeBoxList(o+[0 0 0],[P.W P.L P.H],"SiO2","")];
        bx=[bx; makeBoxList(o+[0 0 P.H],[P.W P.L P.H_Ti_bottom],"Titanium","")];
        bx=[bx; makeBoxList(o+[0 0 P.H+P.H_Ti_bottom],[P.W P.L P.H_gold_bottom],"Gold","")];
        z0_top=o(3)+P.H+P.H_Ti_bottom+P.H_gold_bottom;
        L_vac_eff=min(max(P.L_vac,0),P.L); L_left=max(0,(P.L-L_vac_eff)/2); L_right=max(0,P.L-(L_left+L_vac_eff));
        if L_left>0,  bx=[bx; makeBoxList([o(1) o(2) z0_top],[P.W L_left P.H_vac],"HfO2","")]; end
        if L_vac_eff>0, bx=[bx; makeBoxList([o(1) o(2)+L_left z0_top],[P.W L_vac_eff P.H_vac],"Vacuum","")]; end
        if L_right>0, bx=[bx; makeBoxList([o(1) o(2)+L_left+L_vac_eff z0_top],[P.W L_right P.H_vac],"HfO2","")]; end
        x0_cap=o(1)+(P.W-P.W_Au_top)/2; y0_cap=o(2)+(P.L-P.L_Au_top)/2; z0_cap=z0_top+P.H_vac;
        if P.W_Au_top>0 && P.L_Au_top>0 && P.H_Au_top>0
            bx=[bx; makeBoxList([x0_cap y0_cap z0_cap],[P.W_Au_top P.L_Au_top P.H_Au_top],"Gold",contactName)];
        end
        if P.T_film>0
            bx=[bx; makeBoxList([x0_cap-P.T_film, y0_cap, z0_cap],[P.T_film, P.L_Au_top, P.H_Au_top],"Titanium","")];
            bx=[bx; makeBoxList([x0_cap+P.W_Au_top, y0_cap, z0_cap],[P.T_film, P.L_Au_top, P.H_Au_top],"Titanium","")];
            bx=[bx; makeBoxList([x0_cap-P.T_film, y0_cap-P.T_film, z0_cap],[P.W_Au_top+2*P.T_film, P.T_film, P.H_Au_top],"Titanium","")];
            bx=[bx; makeBoxList([x0_cap-P.T_film, y0_cap+P.L_Au_top, z0_cap],[P.W_Au_top+2*P.T_film, P.T_film, P.H_Au_top],"Titanium","")];
        end
        x_in_min=x0_cap-P.T_film; x_in_max=x0_cap+P.W_Au_top+P.T_film;
        y_in_min=y0_cap-P.T_film; y_in_max=y0_cap+P.L_Au_top+P.T_film;
        T_left_frame =max(0,(x_in_min-o(1)));
        T_right_frame=max(0,(o(1)+P.W-x_in_max));
        T_front_frame=max(0,(y_in_min-o(2)));
        T_back_frame =max(0,(o(2)+P.L-y_in_max));
        z0_frame=z0_cap;
        if T_left_frame>0,  bx=[bx; makeBoxList([o(1) o(2) z0_frame],[T_left_frame  P.L P.H_Au_top],"SiO2","")]; end
        if T_right_frame>0, bx=[bx; makeBoxList([x_in_max o(2) z0_frame],[T_right_frame P.L P.H_Au_top],"SiO2","")]; end
        if T_front_frame>0, bx=[bx; makeBoxList([x_in_min o(2) z0_frame],[max(0,x_in_max-x_in_min) T_front_frame P.H_Au_top],"SiO2","")]; end
        if T_back_frame>0,  bx=[bx; makeBoxList([x_in_min y_in_max z0_frame],[max(0,x_in_max-x_in_min) T_back_frame P.H_Au_top],"SiO2","")]; end
    end

    function bx=boxesCentralSquareMV(P,o,contactName)
        bx=[];
        Lside=P.L; Wside=P.L;
        bx=[bx; makeBoxList(o+[0 0 0],[Lside Wside P.H],"SiO2","")];
        bx=[bx; makeBoxList(o+[0 0 P.H],[Lside Wside P.H_Ti_bottom],"Titanium","")];
        bx=[bx; makeBoxList(o+[0 0 P.H+P.H_Ti_bottom],[Lside Wside P.H_gold_bottom],"Gold","")];
        z0_top=o(3)+P.H+P.H_Ti_bottom+P.H_gold_bottom;
        if P.H_vac>0, bx=[bx; makeBoxList([o(1) o(2) z0_top],[Lside Wside P.H_vac],"Vacuum","")]; end
        capSide=max(0, min(P.CapSide_center, Lside));
        x0_cap=o(1)+(Lside-capSide)/2; y0_cap=o(2)+(Wside-capSide)/2; z0_cap=z0_top+P.H_vac;
        if capSide>0 && P.H_Au_top>0
            bx=[bx; makeBoxList([x0_cap y0_cap z0_cap],[capSide capSide P.H_Au_top],"Gold",contactName)];
        end
        if P.T_film>0
            bx=[bx; makeBoxList([x0_cap-P.T_film, y0_cap, z0_cap],[P.T_film, capSide, P.H_Au_top],"Titanium","")];
            bx=[bx; makeBoxList([x0_cap+capSide,  y0_cap, z0_cap],[P.T_film, capSide, P.H_Au_top],"Titanium","")];
            bx=[bx; makeBoxList([x0_cap-P.T_film, y0_cap-P.T_film, z0_cap],[capSide+2*P.T_film, P.T_film, P.H_Au_top],"Titanium","")];
            bx=[bx; makeBoxList([x0_cap-P.T_film, y0_cap+capSide,  z0_cap],[capSide+2*P.T_film, P.T_film, P.H_Au_top],"Titanium","")];
        end
        x_in_min=x0_cap-P.T_film; x_in_max=x0_cap+capSide+P.T_film;
        y_in_min=y0_cap-P.T_film; y_in_max=y0_cap+capSide+P.T_film;
        T_left_frame =max(0,(x_in_min-o(1)));
        T_right_frame=max(0,(o(1)+Wside-x_in_max));
        T_front_frame=max(0,(y_in_min-o(2)));
        T_back_frame =max(0,(o(2)+Wside-y_in_max));
        if T_left_frame>0,  bx=[bx; makeBoxList([o(1) o(2) z0_cap],[T_left_frame  Wside P.H_Au_top],"SiO2","")]; end
        if T_right_frame>0, bx=[bx; makeBoxList([x_in_max o(2) z0_cap],[T_right_frame Wside P.H_Au_top],"SiO2","")]; end
        if T_front_frame>0, bx=[bx; makeBoxList([x_in_min o(2) z0_cap],[max(0,x_in_max-x_in_min) T_front_frame P.H_Au_top],"SiO2","")]; end
        if T_back_frame>0,  bx=[bx; makeBoxList([x_in_min y_in_max z0_cap],[max(0,x_in_max-x_in_min) T_back_frame P.H_Au_top],"SiO2","")]; end
    end

    function FL=boxesFiller(o,Lx,Wy,hvec)
        FL=[]; z=o(3);
        if Lx>0&&Wy>0&&hvec(1)>0, FL=[FL; makeBoxList([o(1) o(2) z],[Lx Wy hvec(1)],"SiO2","")]; z=z+hvec(1); end
        if Lx>0&&Wy>0&&hvec(2)>0, FL=[FL; makeBoxList([o(1) o(2) z],[Lx Wy hvec(2)],"Gold","")];   z=z+hvec(2); end
        if Lx>0&&Wy>0&&hvec(3)>0, FL=[FL; makeBoxList([o(1) o(2) z],[Lx Wy hvec(3)],"HfO2","")]; z=z+hvec(3); end
        if Lx>0&&Wy>0&&hvec(4)>0, FL=[FL; makeBoxList([o(1) o(2) z],[Lx Wy hvec(4)],"SiO2","")]; end
    end

%% ================== Box helpers ==================
    function bx=makeBoxList(o,sz,mat,rname)
        V=vbox(o,sz(1),sz(2),sz(3));
        if nargin<4, rname=""; end
        bx=struct('V',V,'mat',string(mat),'name',string(rname));
    end
    function V=vbox(o,Lx,Wy,Hz)
        V=[ o;
            o+[Lx 0 0];
            o+[0 Wy 0];
            o+[0 0 Hz];
            o+[Lx Wy 0];
            o+[Lx 0 Hz];
            o+[0 Wy Hz];
            o+[Lx Wy Hz] ];
    end
    function boxesR=rotateBoxesAround(boxes,Cxy,angDeg)
        if isempty(boxes), boxesR=boxes; return; end
        th=deg2rad(angDeg); R2=[cos(th) -sin(th); sin(th) cos(th)];
        boxesR=boxes;
        for ii=1:numel(boxes)
            V=boxes(ii).V; V(:,1)=V(:,1)-Cxy(1); V(:,2)=V(:,2)-Cxy(2);
            V(:,1:2)=(R2*V(:,1:2).').'; V(:,1)=V(:,1)+Cxy(1); V(:,2)=V(:,2)+Cxy(2);
            boxesR(ii).V=V;
        end
    end
    function boxesT=translateBoxesXY(boxes,dxy)
        boxesT=boxes; for ii=1:numel(boxes), V=boxes(ii).V; V(:,1)=V(:,1)+dxy(1); V(:,2)=V(:,2)+dxy(2); boxesT(ii).V=V; end
    end
    function [LL,UR]=boxesAABB(boxes)
        if isempty(boxes), LL=[0 0]; UR=[0 0]; return; end
        allV=vertcat(boxes.V); LL=[min(allV(:,1)) min(allV(:,2))]; UR=[max(allV(:,1)) max(allV(:,2))];
    end

%% ================== Draw (3D) ==================
    function H=drawBoxes(ax3,boxes)
        [c_base,c_ti,c_au,c_vac,~,c_hfo2,edgeCol,lw]=colorsAndStyle_();
        cmap=struct('SiO2',c_base,'Titanium',c_ti,'Gold',c_au,'Vacuum',c_vac,'HfO2',c_hfo2);
        F=faces6(); H=gobjects(0);
        for ii=1:numel(boxes)
            V=boxes(ii).V; 
            if isfield(cmap,boxes(ii).mat), col=cmap.(boxes(ii).mat); else, col=[0.8 0.8 0.8]; end
            fa=strcmp(boxes(ii).mat,'Vacuum')*0.35 + ~strcmp(boxes(ii).mat,'Vacuum')*1.0;
            H=[H; patch(ax3,'Vertices',V,'Faces',F,'FaceColor',col,'FaceAlpha',fa,'EdgeColor',edgeCol,'LineWidth',lw,'FaceLighting','flat','BackFaceLighting','unlit')];
        end
    end
    function [c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeColor,lineWidth]=colorsAndStyle_()
        c_base=[0.55 0.35 0.20]; c_ti=[0.20 0.20 0.20]; c_au=[1.00 0.84 0.00];
        c_vac=[0.00 0.60 0.00]; a_vac=0.35; c_hfo2=[0.80 0.65 0.45]; edgeColor='k'; lineWidth=1.0;
    end
    function F=faces6()
        F=[1 2 5 3; 3 5 8 7; 1 3 7 4; 2 6 8 5; 1 4 6 2; 4 7 8 6];
    end

%% ================== Canvas → Poses ==================
    function Poses=computeCanvasPosesLikeRenderer(Bseq,P,NcellsArm,BusFillX)
        [~,ord]=sort([Bseq.t]); Bseq=Bseq(ord);
        placed=struct('LL',{},'w',{},'hgt',{},'rot',{}); Poses=repmat(struct('type','','rot',0,'LL',[0 0]),1,numel(Bseq));
        for k=1:numel(Bseq)
            b=Bseq(k); [w0,h0]=nativeBlockSize(b,P,NcellsArm,BusFillX);
            if mod(b.rot,180)==0, wnat=w0; hnat=h0; else, wnat=h0; hnat=w0; end
            [hasAnchor,aIdx,side]=findAnchorNeighbor(Bseq,placed,k);
            if ~hasAnchor, LL=[0,0];
            else
                prev=placed(aIdx);
                switch side
                    case 'E', LL=[prev.LL(1)+prev.w, prev.LL(2)];
                    case 'W', LL=[prev.LL(1)-wnat,    prev.LL(2)];
                    case 'N', LL=[prev.LL(1),         prev.LL(2)+prev.hgt];
                    case 'S', LL=[prev.LL(1),         prev.LL(2)-hnat];
                    otherwise, LL=[prev.LL(1)+prev.w, prev.LL(2)];
                end
            end
            Poses(k).type=Bseq(k).type; Poses(k).rot=Bseq(k).rot; Poses(k).LL=LL;
            placed(k).LL=LL; placed(k).w=wnat; placed(k).hgt=hnat; placed(k).rot=Bseq(k).rot;
        end
    end

%% ================== SDE Emit ==================
    function emitHeader(fid)
        fprintf(fid,';; ======================================================\n');
        fprintf(fid,';; SENTARUS SDE — Exported from MATLAB GUI (MV/L/T/Bus)\n');
        fprintf(fid,';; Geometry emitted as explicit cuboids already rotated & placed.\n');
        fprintf(fid,';; Any box with non-empty name becomes a contact region.\n');
        fprintf(fid,';; Date: %s\n',datestr(now,'yyyy-mm-dd HH:MM:SS'));
        fprintf(fid,';; ======================================================\n\n');
        fprintf(fid,'(sde:clear)\n');
        fprintf(fid,'(define (mk-cuboid x1 y1 z1 x2 y2 z2 mat)\n');
        fprintf(fid,'  (sdegeo:create-cuboid (position x1 y1 z1) (position x2 y2 z2) mat))\n\n');
    end

    function emitGlobalParams(fid,P,NcellsArm)
        um2mm = 1e-3;
        fprintf(fid,';; ---- Global parameters (nm) ----\n');
        fprintf(fid,'(sde:define-parameter "L" %.9g)\n',        P.L           * um2mm);
        fprintf(fid,'(sde:define-parameter "W" %.9g)\n',        P.W           * um2mm);
        fprintf(fid,'(sde:define-parameter "H" %.9g)\n',        P.H           * um2mm);
        fprintf(fid,'(sde:define-parameter "H_Ti_bottom" %.9g)\n',  P.H_Ti_bottom * um2mm);
        fprintf(fid,'(sde:define-parameter "H_gold_bottom" %.9g)\n',P.H_gold_bottom* um2mm);
        fprintf(fid,'(sde:define-parameter "H_Au_top" %.9g)\n',    P.H_Au_top    * um2mm);
        fprintf(fid,'(sde:define-parameter "H_vac" %.9g)\n',       P.H_vac       * um2mm);
        fprintf(fid,'(sde:define-parameter "L_vac" %.9g)\n',       P.L_vac       * um2mm);
        fprintf(fid,'(sde:define-parameter "L_Au_top" %.9g)\n',    P.L_Au_top    * um2mm);
        fprintf(fid,'(sde:define-parameter "W_Au_top" %.9g)\n',    P.W_Au_top    * um2mm);
        fprintf(fid,'(sde:define-parameter "T_film" %.9g)\n',      P.T_film      * um2mm);
        fprintf(fid,'(sde:define-parameter "NcellsArm" %d)\n\n',   NcellsArm);
    end

    function emitContactSetsAndAssignments(fid, ContactMap)
        keys = ContactMap.keys;
        if isempty(keys)
            fprintf(fid,';; [INFO] No named contact regions found.\n');
            return;
        end
        pal = [ 0.00 0.447 0.741;
                0.850 0.325 0.098;
                0.929 0.694 0.125;
                0.494 0.184 0.556;
                0.466 0.674 0.188;
                0.301 0.745 0.933 ];
        np = size(pal,1);
        for i=1:numel(keys)
            cname = keys{i};
            col = pal(mod(i-1,np)+1,:);
            fprintf(fid,'\n(sdegeo:define-contact-set "%s" 4 (color:rgb %.3f %.3f %.3f) "##")\n', ...
                cname, col(1), col(2), col(3));
            fprintf(fid,'(sdegeo:set-current-contact-set "%s")\n', cname);
            vars = ContactMap(cname);
            for v=1:numel(vars)
                fprintf(fid,'(sdegeo:set-contact %s "%s")\n', vars{v}, cname);
            end
        end
    end

    function emitMeshRefAndBuild(fid)
        fprintf(fid,'(sdedr:define-refinement-function "RFn" "MaxLenInt" "Gold" "Vacuum"   0.0001 "DoubleSide")\n');
        fprintf(fid,'(sdedr:define-refinement-function "RFn" "MaxLenInt" "Gold" "Titanium" 0.0001)\n\n');
        fprintf(fid,'(sde:build-mesh "n@node@")\n');
    end

%% ================== Helpers for SDevice export ==================
    function names = collectContactNames(Poses, P, NcellsArm, BusFillX)
        present = string.empty(0,1);
        for k=1:numel(Poses)
            t=lower(string(Poses(k).type));
            [boxes,nativeLL,nativeSize]=buildBlockBoxesNative(t,P,NcellsArm,BusFillX); 
            Cn=nativeLL+nativeSize/2;
            boxesR=rotateBoxesAround(boxes,Cn,Poses(k).rot);
            [minxyR,~]=boxesAABB(boxesR); delta=[Poses(k).LL(1)-minxyR(1), Poses(k).LL(2)-minxyR(2)];
            boxesT=translateBoxesXY(boxesR,delta);
            for ii=1:numel(boxesT)
                nm=strtrim(string(boxesT(ii).name));
                if strlength(nm)>0
                    present(end+1,1) = nm; 
                end
            end
        end
        present = unique(present);

        names = struct();
        names.bottom = "BottomContact";

        names.center = present(contains(present,"_X"));

        inL  = sortContactsByPrefix(present,"TopContact_L_IN_");
        inT  = sortContactsByPrefix(present,"TopContact_T_IN_");
        inMV = sortContactsByPrefix(present,"TopContact_MV_IN_");
        names.south = [inL; inT; inMV];

        outL  = sortContactsByPrefix(present,"TopContact_L_OUT_");
        outT  = sortContactsByPrefix(present,"TopContact_T_OUT_");
        outMV = sortContactsByPrefix(present,"TopContact_MV_OUT_");
        names.east = [outL; outT; outMV];

        names.north = string.empty(0,1);
        names.west  = string.empty(0,1);

        names.bus   = sortContactsByPrefix(present,"TopContact_b_");
    end

    function arr = sortContactsByPrefix(allNames,prefix)
        arr = allNames(startsWith(allNames,prefix));
        if isempty(arr), return; end
        suffix = extractAfter(arr, strlength(prefix));
        idx = str2double(suffix);
        nanmask = isnan(idx);
        if any(nanmask), idx(nanmask) = 1:nnz(nanmask); end
        [~,ord] = sort(idx);
        arr = arr(ord);
    end

    function emitSDevice(fid, CN)
        % ==== HEADER: File, Electrode, Physics, Thermode, Plot, Math ====
        fprintf(fid,'File {\n');
        fprintf(fid,'  Grid   \t= "n1_msh.tdr"\n');
        fprintf(fid,'  Plot   \t= "n@node@_clock_des.tdr"\n');
        fprintf(fid,'  Current   = "n@node@_clock_des.plt"\n');
        fprintf(fid,'  Parameter = "sdevice.par"\n');
        fprintf(fid,'}\n\n');

        fprintf(fid,'Electrode {\n\n');
        fprintf(fid,'*BOTTOMCONTACT\n\n');
        fprintf(fid,'  { Name = "%s"                \tVoltage = 0.0 }\n\n', CN.bottom);

        fprintf(fid,'*INPUTS SEQUENCES TOPCONTACT\n\n');
        printContactList(fid,CN.south);
        printContactList(fid,CN.north);
        printContactList(fid,CN.west);
        printContactList(fid,CN.bus);

        fprintf(fid,'*CENTER CELL TOPCONTACT\n\n');
        if ~isempty(CN.center)
            fprintf(fid,'  { Name = "%s"\tVoltage = 0.0 }\n\n', CN.center);
        end

        fprintf(fid,'*OUTPUT TOPCONTACT\n\n');
        printContactList(fid,CN.east);

        fprintf(fid,'\n}\n\n');

        fprintf(fid,'# ----------------------------\n# PHYSICS\n# ----------------------------\n');
        fprintf(fid,'Physics (Material="HfO2") {\n  CondInsulator\n}\n\n');
        fprintf(fid,'Physics (Material="SiO2") {\n  CondInsulator\n}\n\n');

        fprintf(fid,'Thermode {\n\n');
        writeThermodes(fid,CN.south);
        writeThermodes(fid,CN.north);
        writeThermodes(fid,CN.west);
        writeThermodes(fid,CN.bus);
        if ~isempty(CN.center)
            fprintf(fid,'  { Name = "%s"                \tTemperature = 300 SurfaceResistance = 1e-5 }\n\n', CN.center);
        end
        writeThermodes(fid,CN.east);
        fprintf(fid,'  { Name = "%s"              \tTemperature = 300 SurfaceResistance = 1e-5 }\n', CN.bottom);
        fprintf(fid,'}\n\n');

        fprintf(fid,'# ----------------------------\n# PLOT\n# ----------------------------\n');
        fprintf(fid,'Plot {\n');
        fprintf(fid,'  Potential\n  ElectricField\n  DielectricConstant\n  Temperature\n');
        fprintf(fid,'  ConductionCurrentDensity/Vector\n  DisplacementCurrentDensity/Vector\n');
        fprintf(fid,'  TotalCurrent/Vector\n  SpaceCharge\n  Potential Doping\n');
        fprintf(fid,'  BandGap ElectronAffinity\n  ConductionBandEnergy ValenceBandEnergy\n');
        fprintf(fid,'}\n\n');

        fprintf(fid,'Math {\n  RelErrControl\n  Extrapolate\n}\n\n');

        fprintf(fid,'Solve {\n');
        fprintf(fid,'  Coupled (Iterations= 100 LineSearchDamping= 1e-8) {Poisson}\n');
        fprintf(fid,'  Coupled{ Poisson Temperature Contact CondInsulator }\n');
        fprintf(fid,'  Plot(FilePrefix="n@node@_equilibrium")\n\n');

        % ==== 8 CONFIGURATIONS (Bus + L-wire mapping) ====
        % CN.bus: TopContact_b_1..3 (ascending index)
        % CN.east: TopContact_L_OUT_1..4
        % CN.south: TopContact_L_IN_1..4
        % CN.center: TopContact_L_X

        % cfg1: NLH
        emitConfigBlock(fid, CN, "cfg1", "NLH", ...
            [ 3 -3  0], ...  % bus (b1..b3)
            [ 0  3 -3  0], ... % OUT_1..4
            -3, ...          % center
            [ 3 -3  0  3]);  % IN_1..4
        emitReturnToZero(fid, CN);

        % cfg2: HLN
        emitConfigBlock(fid, CN, "cfg2", "HLN", ...
            [ 0 -3  3], ...
            [ 3  0 -3  3], ...
            -3, ...
            [ 0 -3  3  0]);
        emitReturnToZero(fid, CN);

        % cfg3: HNL
        emitConfigBlock(fid, CN, "cfg3", "HNL", ...
            [-3  0  3], ...
            [ 3 -3  0  3], ...
             0, ...
            [-3  0  3 -3]);
        emitReturnToZero(fid, CN);

        % cfg4: HHL
        emitConfigBlock(fid, CN, "cfg4", "HHL", ...
            [-3  3  3], ...
            [ 3 -3  3  3], ...
             3, ...
            [-3  3  3 -3]);
        emitReturnToZero(fid, CN);

        % cfg5: NHL
        emitConfigBlock(fid, CN, "cfg5", "NHL", ...
            [-3  3  0], ...
            [ 0 -3  3  0], ...
             3, ...
            [-3  3  0 -3]);
        emitReturnToZero(fid, CN);

        % cfg6: LHN
        emitConfigBlock(fid, CN, "cfg6", "LHN", ...
            [ 0  3 -3], ...
            [-3  0  3 -3], ...
             3, ...
            [ 0  3 -3  0]);
        emitReturnToZero(fid, CN);

        % cfg7: LNH
        emitConfigBlock(fid, CN, "cfg7", "LNH", ...
            [ 3  0 -3], ...
            [-3  3  0 -3], ...
             0, ...
            [ 3  0 -3  3]);
        emitReturnToZero(fid, CN);

        % cfg8: LLH
        emitConfigBlock(fid, CN, "cfg8", "LLH", ...
            [ 3 -3 -3], ...
            [-3  3 -3 -3], ...
            -3, ...
            [ 3 -3 -3  3]);
        emitReturnToZero(fid, CN);

        fprintf(fid,'}\n'); % end Solve
    end

    function printContactList(fid,names)
        for i=1:numel(names)
            fprintf(fid,'  { Name = "%s"   Voltage = 0.0 }\n', names(i));
        end
        if ~isempty(names), fprintf(fid,'\n'); end
    end

    function writeThermodes(fid,names)
        for i=1:numel(names)
            fprintf(fid,'  { Name = "%s"  Temperature = 300 SurfaceResistance = 1e-5 }\n', names(i));
        end
    end

    function emitConfigBlock(fid, CN, cfgName, cfgLabel, pattBus, pattOut, centerV, pattIn)
        fprintf(fid,'  # =====================================================================\n');
        fprintf(fid,'  # ==== CONFIG (%s) %s ====\n', cfgName, cfgLabel);
        fprintf(fid,'  quasistationary (InitialStep = 0.01 Increment = 1.5 MaxStep = 0.05 MinStep=1e-4\n\n');

        fprintf(fid,'  Goal{ name = "%s"                \tvoltage = 0.0 }\n\n', CN.bottom);

        % Bus contacts (TopContact_b_1..)
        writeGoalsWithPatternSimple(fid, CN.bus, pattBus);

        % Outputs (L_OUT_1..4, plus any other east contacts)
        writeGoalsWithPatternSimple(fid, CN.east, pattOut);

        % Center contact(s), if present
        if ~isempty(CN.center)
            fprintf(fid,'  Goal{ name = "%s"  \tvoltage = %.1f }\n\n', CN.center, centerV);
        end

        % Inputs (L_IN_1..4 + any additional IN from south/north/west)
        writeGoalsWithPatternSimple(fid, CN.south, pattIn);
        writeGoalsWithPatternSimple(fid, CN.north, pattIn);
        writeGoalsWithPatternSimple(fid, CN.west,  pattIn);

        fprintf(fid,'\tplot { range=(0, 1) intervals= 1}\n');
        fprintf(fid,'  ){coupled { Poisson Temperature CondInsulator }}\n\n');
        fprintf(fid,'  Plot(FilePrefix="n@node@_%s")\n\n', cfgName);
    end

    function emitReturnToZero(fid, CN)
        fprintf(fid,'  # ---- return to 0 V on all top contacts\n');
        fprintf(fid,'  quasistationary (InitialStep = 0.01 Increment = 1.5 MaxStep = 0.05 MinStep=1e-4\n\n');
        fprintf(fid,'  Goal{ name = "%s"                \tvoltage = 0.0 }\n\n', CN.bottom);

        writeGoalsToZero(fid, CN.bus);
        writeGoalsToZero(fid, CN.east);
        writeGoalsToZero(fid, CN.center);
        writeGoalsToZero(fid, CN.south);
        writeGoalsToZero(fid, CN.north);
        writeGoalsToZero(fid, CN.west);

        fprintf(fid,'\tplot { range=(0, 1) intervals= 1}\n');
        fprintf(fid,'  ){coupled { Poisson Temperature CondInsulator }}\n\n');
    end

    function writeGoalsWithPatternSimple(fid, names, patt)
        if isempty(names) || isempty(patt), return; end
        for i=1:numel(names)
            v = patt( min(i, numel(patt)) );
            fprintf(fid,'  Goal{ name = "%s"\tvoltage = %.1f }\n', names(i), v);
        end
        if ~isempty(names), fprintf(fid,'\n'); end
    end

    function writeGoalsToZero(fid, names)
        if isstring(names) || ischar(names)
            % Single center contact
            if strlength(string(names))>0
                fprintf(fid,'  Goal{ name = "%s"\tvoltage = 0.0 }\n\n', string(names));
            end
            return;
        end
        for i=1:numel(names)
            fprintf(fid,'  Goal{ name = "%s"\tvoltage = 0.0 }\n', names(i));
        end
        if ~isempty(names), fprintf(fid,'\n'); end
    end
end
